package org.gualdi.grails.plugins.cms.domain

import org.grails.comments.Commentable
import org.grails.taggable.Taggable
import org.gualdi.grails.plugins.cms.commands.CommandProcessor

class Page implements Taggable, Commentable, Serializable {
  static belongsTo = [category: Category]

  String tag
  String title
  String description
  String image
  String body
  String commands
  Boolean hidden = false
  Boolean preprocess = false
  Boolean commentEnabled = false
  Integer ordine = 0
  Date dateCreated
  Date lastUpdated

  def beforeDelete() {
    // Delete all associated comments
    Page.withNewSession {
      comments?.each { c ->
        removeComment(c)
      }
    }
  }

  def getSite() {
    return getSiteImpl(this?.category)
  }

  private getSiteImpl(category) {
    def tmp = category
    def site = tmp.site
    while (!site) {
      tmp = tmp.parent
      site = tmp.site
    }

    return site
  }

  def getVisible() {
    return !hidden
  }

  def isVisibleForRoles(activeUserRoles) {
    def result = true
    def item = this?.category
    while (item && result) {
      def categoryRole = item.role != null && item.role != 'null' ? item.role : null
      if (categoryRole) {
        result = categoryRole in activeUserRoles
      }
      item = item.parent
    }
    return result
  }

  def hasCommands() {
    return this.commands?.trim()?.size() > 0
  }

  static constraints = {
    category(nullable: false)

    tag(blank: true, nullable: true, maxSize: 50, matches: /[a-z0-9_-]*/, validator: { value, obj ->
      if (value) {
        // Verifica il rispetto dell'unicita' del tag nel sito
        // Ottiene il sito di appartenenza
        def s = obj.getSite()
        if (s.pages.any { p -> (p.tag == value) && (obj.id ? (p.id != obj.id) : true) }) {
          return ['page.error.tagMustBeUnique']
        }
      }
    })

    hidden(nullable: true, blank: true)
    preprocess(nullable: true, blank: true)
    title(blank: false, maxSize: 255)
    description(nullable: true, blank: true, maxSize: 255)
    image(nullable: true, blank: true, maxSize: 255)
    ordine(nullable: true, blank: false)
    body(blank: false)

    commands(blank: true, nullable: true, validator: { value, obj ->
      if (value) {
        def cmd = new CommandProcessor()
        if (!cmd.process(value)) {
          return ['page.error.invalidCommand']
        }
      }
    })

    dateCreated(nullable: true)
    lastUpdated(nullable: true)
  }

  static mapping = {
    columns {
      body type: 'text'
    }
  }

  // todo verificare se site viene considerato
  static searchable = {
    only = ["title", "body", "site"]
  }
}