package org.gualdi.plugins.cms

import org.gualdi.grails.plugins.cms.domain.*
import grails.plugin.cache.Cacheable
import grails.plugin.cache.CacheEvict
import org.springframework.cache.annotation.Caching
import org.springframework.transaction.annotation.Transactional

class CmsBaseService {

    static transactional = false

    def grailsApplication
    def grailsCacheManager
    def cmsSecurityService

    // =====================================================================
    // Internal functions
    // =====================================================================

    /**
     * Scan all mixins and return a data structure representing the menu
     */
    @Cacheable('ixcms-internal')
    def composeAdminMenuFromMixins() {
        def finalMenu = [:]

        grailsApplication.controllerMixinClasses.each { mixinArtefact ->
            if (mixinArtefact.clazz.metaClass.hasProperty(mixinArtefact.clazz, 'menuEntry')) {
                def menuDef = mixinArtefact.clazz.menuEntry
                finalMenu["${menuDef.key}"] = menuDef
            }
        }

        return finalMenu
    }

    // =====================================================================
    // Internal functions for cache
    // =====================================================================
    private void clearUploadsCache() {
        grailsCacheManager.getCache("ixcms-uploads").clear()
    }

    // =====================================================================
    // Tags support
    // =====================================================================

    def getTagsFor(clazz, term) {
        def result = []
        if (term) {
            result = clazz.allTags.findAll { it.startsWith(term) }
        }
        else {
            result = clazz.allTags
        }

        return result
    }

    // =====================================================================
    // Sites management
    // =====================================================================

    Site getSite(Long id) {
        Site.get(id)
    }

    def listSites() {
        Site.list(sort: "tag", order: "asc")
    }

    @Transactional
    def saveSite(cmd) {
        def site = new Site(cmd.properties)
        site.save()
    }

    @Transactional
    def updateSite(cmd) {
        def site = getSite(cmd.id)
        site.properties = cmd.properties
        site.save()
    }

    @Transactional
    def deleteSite(Long id) {
        def site = getSite(id)
        site.delete()
    }

    Boolean isSiteTagUnique(String sitetag, Long id) {
        Site site = Site.findByTag(sitetag)
        def result = !(site && (id ? (site.id != id) : true))

        return result
    }

    String getGoogleAnalyticsCode(String sitetag) {
        def site = Site.findByTag(sitetag)
        def code = site?.googleTag?.trim()
        return code
    }

    // =====================================================================
    // Pages management
    // =====================================================================

    def getPage(Long id) {
        Page.get(id)
    }

    def getPage(String sitetag, String id) {
        Site site = Site.findByTag(sitetag)
        site.getPageByTag(id)
    }

    def listTaggedPages(String sitetag) {
        def result = []
        if (sitetag) {
            Site site = Site.findByTag(sitetag)
            result = site.pages.findAll { p -> p.tag != null }.collect { p ->
                [id: p.id, title: p.title, tag: p.tag, hidden: p.hidden]
            }
        }

        return result
    }

    @Transactional
    @CacheEvict(value='ixcms-structure', allEntries=true)
    def savePage(cmd, String sitetag) {
        Site site = Site.findByTag(sitetag)

        def page = new Page(cmd.properties)
        page.save(flush: true)

        page.setTags(cmd.tagsAsArray)
    }

    @Transactional
    @CacheEvict(value='ixcms-structure', allEntries=true)
    def updatePage(cmd) {
        def page = getPage(cmd.id)
        page.properties = cmd.properties
        page.save(flush: true)

        page.setTags(cmd.tagsAsArray)
    }

    @Transactional
    @CacheEvict(value='ixcms-structure', allEntries=true)
    def deletePage(Long id) {
        def page = getPage(id)
        page.delete()
    }

    def getCategory(Long id) {
        Category.get(id)
    }

    @Transactional
    @CacheEvict(value='ixcms-structure', allEntries=true)
    def saveCategory(cmd, String sitetag) {
        def category = new Category(cmd.properties)

        // Site must be set only in root categories
        if (!category.parent) {
            Site site = Site.findByTag(sitetag)
            category.site = site
            category.parent = null
        }
        else {
            category.site = null
        }

        category.save()
    }

    @Transactional
    @CacheEvict(value='ixcms-structure', allEntries=true)
    def updateCategory(cmd, String sitetag) {
        def category = getCategory(cmd.id)
        category.properties = cmd.properties

        // Site must be set only in root categories
        if (!category.parent) {
            Site site = Site.findByTag(sitetag)
            category.site = site
            category.parent = null
        }
        else {
            category.site = null
        }

        category.save()

        clearUploadsCache()
    }

    @Transactional
    @CacheEvict(value='ixcms-structure', allEntries=true)
    def deleteCategory(Long id) {
        def category = getCategory(id)
        category.delete()

        clearUploadsCache()
    }

    Boolean isPageTagUnique(String tag, String sitetag, Long id) {
        Boolean result = true
        if (tag) {
            Site site = Site.findByTag(sitetag)
            if (site.pages.any { p -> (p.tag == tag) && (id ? (p.id != id) : true) }) {
                result = false
            }
        }
        return result
    }

    Boolean isProtectionTagUnique(String tag, String sitetag, Long id) {
        Boolean result = true
        if (tag) {
            Site site = Site.findByTag(sitetag)
            if (site.categoriesFlat.any { p -> (p.protectionTag == tag) && (id ? (p.id != id) : true) }) {
                result = false
            }
        }
        return result
    }

    def getSiteObject(Long id, String type) {
        def result = [:]

        result.id = id
        result.type = type

        if (type == 'category') {
            def category = Category.get(id)
            result.name = category.name
            result.hidden = category.hidden
            result.order = category.ordine
            result.pageTag = category.pageTag
            result.role = category.role
        }
        else if (type == 'page') {
            def page = Page.get(id)
            result.tag = page.tag
            result.name = page.title
            result.hidden = page.hidden
            result.order = page.ordine
            result.commentable = page.commentEnabled
            result.preprocessable = page.preprocess
            result.tags = page.tags
        }

        return result
    }

    // Cacheable version. rolesKey is a key contructed summing up the user's roles
    @Cacheable(value='ixcms-structure', key='#rolesKey')
    def getSiteStructureForCurrentUser(String sitetag, String rolesKey) {
        def structure = getSiteStructure(sitetag)
        def activeUserRoles = cmsSecurityService.currentUserRoles

        disableUnauthorizedNodes(structure, activeUserRoles)

        return structure
    }

    def getSiteStructureForCurrentUser(String sitetag) {
        def structure = getSiteStructure(sitetag)
        def activeUserRoles = cmsSecurityService.currentUserRoles

        disableUnauthorizedNodes(structure, activeUserRoles)

        return structure
    }

    private disableUnauthorizedNodes(structure, activeUserRoles) {
        for (item in structure) {
            def isVisibleToUser = true

            // Compatibility with domain managed by the old ixcms
            def categoryRole = item.role != null && item.role != 'null' ? item.role : null
            if (categoryRole) {
                isVisibleToUser = categoryRole in activeUserRoles
            }

            item.hidden = item.hidden ?: !isVisibleToUser

            if (!item.hidden) {
                if (item.type == 'category' && item?.children?.size() > 0 && item?.children?.any { !it.hidden }) {
                    disableUnauthorizedNodes(item.children, activeUserRoles)
                }
            }
        }
    }

    @Cacheable('ixcms-structure')
    def getSiteStructure(String sitetag) {
        def result = []

        Site.withNewSession {
            Site site = Site.findByTag(sitetag)

            site?.sortedCategories?.each { cat ->
                if (cat.categories || cat.pages) {
                    getSiteStructureImpl(site.tag, cat, 1, result)
                }
                else {
                    result << [id: "c-${cat.id}", name: cat.name, hidden: cat.hidden, pageTag: cat.pageTag, type: 'category', role: cat.role, objId: cat.id]
                }
            }
        }

        return result
    }

    private getSiteStructureImpl(sitetag, category, level, result) {
        def itm = [id: "c-${category.id}", name: category.name, hidden: category.hidden, pageTag: category.pageTag, type: 'category', role: category.role, objId: category.id, children: []]

        result << itm

        def allItems = category?.sortedCategories + category?.sortedPages
        allItems.sort { it.ordine }.each { v ->
            if (v instanceof org.gualdi.grails.plugins.cms.domain.Category) {
                getSiteStructureImpl(sitetag, v, level + 1, itm.children)
            }
            else {
                itm.children << [id: "p-${v.id}", name: v.title, tag: v.tag, hidden: v.hidden, type: 'page', objId: v.id]
            }
        }
    }

    List listCategoriesFlat(String sitetag, Long excludedId = 0) {
        List result = []

        Site site = Site.findByTag(sitetag)

        Site.withTransaction {
            site?.sortedCategories?.each { cat ->
                listCategoriesFlatImpl(site.tag, cat, 0, result, excludedId)
            }
        }
        return result
    }

    private listCategoriesFlatImpl(String sitetag, Category category, int level, List result, Long excludedId) {
        if (category.id != excludedId) {
            result << [id: category.id, name: category.name, level: level, htmlName: ("--" * level) + " " + category.name]

            category?.sortedCategories?.each { v ->
                listCategoriesFlatImpl(sitetag, v, level + 1, result, excludedId)
            }
        }
    }

    @Transactional
    @CacheEvict(value='ixcms-structure', allEntries=true)
    Boolean moveObject(Map info) {
        Boolean result = false

        // Define target category
        Category target = null
        if (info.target.type == 'category') {
            target = getCategory(info.target.objId)
        }
        else {
            def page = getPage(info.target.objId)
            target = page.category
        }

        // Move the object
        def source
        if (info.source.type == 'category') {
            source = getCategory(info.source.objId)

            if (info.target.type == 'category' && info.target.position == 'inside') {
                source.site = null
                source.parent = target
                result = true
            }
            else if (info.target.type == 'page' && info.target.position in ['after', 'before']) {
                source.site = null
                source.parent = target
                result = true
            }
            else if (info.target.type == 'category' && info.target.position in ['after', 'before']) {
                if (target.parent == null) {
                    source.site = target.site
                    source.parent = null
                    result = true
                }
                else {
                    source.site = null
                    source.parent = target.parent
                    result = true
                }
            }
        }
        else if (info.source.type == 'page') {
            source = getPage(info.source.objId)
            source.category = target
            result = true
        }

        if (result) {
            source.save()

            // Fix objects order
            info.objectsOrder.each { itm ->
                def obj = itm.type == 'category' ? getCategory(itm.id) : getPage(itm.id)
                obj.ordine = itm.order
                obj.save()
            }

            clearUploadsCache()
        }

        return result
    }

    // =====================================================================
    // News management
    // =====================================================================

    NewsItem getNews(Long id) {
        NewsItem.get(id)
    }

    def listNews(String sitetag, Map params) {
        Site site = Site.findByTag(sitetag)

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def c = NewsItem.createCriteria()
        def results = c.list(max: params.max, offset: params.offset) {
            eq("site", site)
            order("date", "desc")
        }

        results
    }

    @Transactional
    def saveNews(cmd, String sitetag) {
        Site site = Site.findByTag(sitetag)

        def news = new NewsItem(cmd.properties)
        news.site = site
        news.save(flush: true)

        news.setTags(cmd.tagsAsArray)
    }

    @Transactional
    def updateNews(cmd) {
        def news = getNews(cmd.id)
        news.properties = cmd.properties
        news.save(flush: true)

        news.setTags(cmd.tagsAsArray)
    }

    @Transactional
    def deleteNews(Long id) {
        def news = getNews(id)
        news.delete()
    }

    // =====================================================================
    // Banners management
    // =====================================================================

    Banner getBanner(Long id) {
        Banner.get(id)
    }

    Banner getBannerByTag(String sitetag, String tag) {
        def site = Site.findByTag(sitetag)
        def banner = Banner.findBySiteAndTag(site, tag)

        return banner
    }

    def listBanners(String sitetag, Map params) {
        Site site = Site.findByTag(sitetag)

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def c = Banner.createCriteria()
        def results = c.list(max: params.max, offset: params.offset) {
            eq("site", site)
            order("position", "asc")
            order("ordine", "asc")
        }

        results
    }

    List<Banner> listBannersByPosition(String sitetag, String pos, Map params) {
        params.max = Math.min(params?.max?.toInteger() ?: 8, 100)
        params.offset = params?.offset?.toInteger() ?: 0
        params.sort = params?.sort ?: "ordine"
        params.order = params?.order ?: "asc"

        def site = Site.findByTag(sitetag)

        def banners = Banner.createCriteria().list(
                max: params.max,
                offset: params.offset,
                sort: params.sort,
                order: params.order) {
            eq('site', site)
            eq('position', pos)
            eq('active', true)
        }

        return banners
    }

    @Transactional
    def saveBanner(cmd, String sitetag) {
        Site site = Site.findByTag(sitetag)

        def banner = new Banner(cmd.properties)
        banner.site = site
        banner.save(flush: true)
    }

    @Transactional
    def updateBanner(cmd) {
        def banner = getBanner(cmd.id)
        banner.properties = cmd.properties
        banner.save(flush: true)
    }

    @Transactional
    def deleteBanner(Long id) {
        def banner = getBanner(id)
        banner.delete()
    }

    Boolean isBannerTagUnique(String tag, String sitetag, Long id) {
        Boolean result = true
        if (tag) {
            Site site = Site.findByTag(sitetag)
            def banners = Banner.findAllBySite(site)
            if (banners.any { b -> (b.tag == tag) && (id ? (b.id != id) : true) }) {
                result = false
            }
        }
        return result
    }
}
