package org.gualdi.grails.plugins.cms

import org.codehaus.groovy.grails.plugins.springsecurity.SpringSecurityUtils
import org.springframework.transaction.annotation.Transactional
import org.springframework.dao.DataIntegrityViolationException
import org.gualdi.grails.plugins.cms.exceptions.*
import org.gualdi.grails.plugins.cms.domain.*
import grails.plugin.jxl.builder.ExcelBuilder

class CmsBaseUsersService {

    static transactional = false

    def grailsApplication
    def cmsSecurityService

    // =====================================================================
    // Roles
    // =====================================================================

    def listRoles(Map params) {
        def sitetag = params.sitetag

        def roleClass = lookupRoleClass()

        def userLookup = SpringSecurityUtils.securityConfig.userLookup
        String nameFieldName = SpringSecurityUtils.securityConfig.authority.nameField

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def r = roleClass.createCriteria()
        def roles = r.list(max: params.max, offset: params.offset) {
            order(nameFieldName, "asc")
        }

        return roles
    }

    def listAllRoles(String query = null, Boolean excludeAdmins = false) {
        def r = lookupRoleClass().createCriteria()
        def roles = r.list {
        	and {
	        	if (excludeAdmins) {
	        		ne("authority", "ROLE_ADMIN")
	        	}
	        	if (query) {
        			ilike("authority", "%${query}%")
	        	}
        	}

            order("authority", "asc")
        }

        return roles
    }

    def getRole(Long id) {
        lookupRoleClass().get(id)
    }

    @Transactional
    def saveRole(role, sitetag) {
        def roleClass = lookupRoleClass()

        def r = roleClass.newInstance()
        r.authority = role.authority
        r.save(flash: true)
    }

    @Transactional
    def updateRole(role, sitetag) {
        def roleClass = lookupRoleClass()

        def r = roleClass.get(role.id)
        r.authority = user.authority
        r.save(flush: true)
    }

    @Transactional
    def deleteRole(Long id, String sitetag) {
        def r = lookupRoleClass().get(id)

        try {
            def usedRoles = ['ROLE_ADMIN', 'ROLE_USER']

            def site = Site.findByTag(sitetag)
            def usedRolesByCategories = site.getCategoriesFlat().role
            def usedRolesByUsers = lookupUserRoleClass().findAll().role.authority

            usedRoles << usedRolesByCategories
            usedRoles << usedRolesByUsers
            
            usedRoles = usedRoles.flatten().unique() - null

            if (!(r?.authority in usedRoles)) {
                r.delete(flush: true)
            }
            else {
                throw new CannotDeleteRoleException()
            }
        }
        catch (DataIntegrityViolationException dive) {
            throw new CannotDeleteRoleException(dive)
        }
    }

    // =====================================================================
    // Users
    // =====================================================================
    
    def isUsernameUnique(username, userId) {
        def userClass = lookupUserClass()
        def userLookup = SpringSecurityUtils.securityConfig.userLookup
        String usernameFieldName = userLookup.usernamePropertyName

        def users = userClass.withCriteria {
            and {
                eq(usernameFieldName, username)
                if (userId) {
                    ne('id', userId)
                }
            }
        }
        return users.empty
    }

    def listUsers(Map params) {
        def sitetag = params.sitetag

        def q = params.q
        q = "%" + q + "%"

        def userClass = lookupUserClass()

        def userLookup = SpringSecurityUtils.securityConfig.userLookup
        String nameFieldName = userLookup.namePropertyName
        String surnameFieldName = userLookup.surnamePropertyName
        String usernameFieldName = userLookup.usernamePropertyName

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def u = userClass.createCriteria()
        def users = u.list(max: params.max, offset: params.offset) {

            or {
                if (params.q) {
                    ilike(usernameFieldName, q)
                    ilike(nameFieldName, q)
                    ilike(surnameFieldName, q)
                }
            }

            order(surnameFieldName, "asc")
            order(nameFieldName, "asc")
        }

        return users
    }

    def listAllUsers(String query) {
        def userClass = lookupUserClass()

        def userLookup = SpringSecurityUtils.securityConfig.userLookup
        String nameFieldName = userLookup.namePropertyName
        String surnameFieldName = userLookup.surnamePropertyName
        String usernameFieldName = userLookup.usernamePropertyName

        def u = userClass.createCriteria()
        def users = u.list {
        	if (query) {
        		or {
        			ilike(nameFieldName, "%${query}%")
        			ilike(surnameFieldName, "%${query}%")
        			ilike(usernameFieldName, "%${query}%")
				}
        	}
            order(nameFieldName, "asc")
            order(surnameFieldName, "asc")
            order(usernameFieldName, "asc")
        }

        return users
    }

    def getUser(Long id) {
        lookupUserClass().get(id)
    }

    @Transactional
    def assignRolesToUsers(roles, users, sitetag) {
        def userClass = lookupUserClass()
        def roleClass = lookupRoleClass()
        def userRoleClass = lookupUserRoleClass()
        def userId
        def roleId
        def user
        def role

		users.each { u ->
			userId = u as Long
			user = userClass.get(userId)
	    	roles.each { r -> 
	    		roleId = r as Long
	    		role = roleClass.get(roleId)
	    		if(!userRoleClass.get(userId, roleId)) {
	    			userRoleClass.create user, role		
	    			log.debug "assegno ruolo " + role + " a utente " + user
	    		}
	    	}
    	}
    }

    @Transactional
    def saveUser(user, sitetag) {
        def userClass = lookupUserClass()
        def roleClass = lookupRoleClass()
        def userRoleClass = lookupUserRoleClass()

        def u = userClass.newInstance()
        u.name = user.name
        u.surname = user.surname
        u.username = user.username
        u.email = user.email
        u.password = user.password
        u.enabled = user.enabled
        u.accountLocked = user.accountLocked
        u.accountExpired = user.accountExpired
        u.passwordExpired = user.passwordExpired
        u.save(flush: true)

        def role
        user?.roles?.each { r ->
            role = roleClass.findByAuthority(r)
            userRoleClass.create u, role, true
        }
    }

    @Transactional
    def updateUser(user, sitetag) {
        def userClass = lookupUserClass()
        def roleClass = lookupRoleClass()
        def userRoleClass = lookupUserRoleClass()

        def u = userClass.get(user.id)
        u.name = user.name
        u.surname = user.surname
        u.username = user.username
        u.email = user.email
        if (user.password) {
            u.password = user.password
        }
        u.enabled = user.enabled
        u.accountLocked = user.accountLocked
        u.accountExpired = user.accountExpired
        u.passwordExpired = user.passwordExpired
        u.save(flush: true)

        // Update roles
        if (user?.roles) {
            def newRoles = user?.roles - u?.authorities?.authority

            // Remove roles
            u?.authorities?.each { r ->
                if (!(r.authority in user.roles)) {
                    userRoleClass.remove u, r, true
                }
            }

            // Add new roles
            def role
            newRoles.each { r ->
                role = roleClass.findByAuthority(r)
                userRoleClass.create u, role, true
            }
        }
        else {
            // Remove all roles
            userRoleClass.removeAll(u)
        }
    }

    @Transactional
    def deleteUser(Long id, String sitetag) {
        def u = lookupUserClass().get(id)
        try {
        	u.delete(flush: true)
        }
        catch (DataIntegrityViolationException dive) {
            throw new CannotDeleteUserException(dive)
        }
    }

    File exportUsers() {
        def userClass = lookupUserClass()

        def userLookup = SpringSecurityUtils.securityConfig.userLookup
        String nameFieldName = userLookup.namePropertyName
        String surnameFieldName = userLookup.surnamePropertyName
        String usernameFieldName = userLookup.usernamePropertyName

        def u = userClass.createCriteria()
        def list = u.list {
            order(surnameFieldName, "desc")
            order(nameFieldName, "desc")
        }

        File ftmp = null
        if (list.size() > 0) {
            ftmp = File.createTempFile("export-", ".xls")
            ftmp.deleteOnExit() //in case something goes wrong and you don't catch it

            def header = [
                    'COGNOME', 'NOME', 'USERNAME', 'EMAIL',
            ]

            new ExcelBuilder().workbook(ftmp.absolutePath) {
                sheet('Users') {

                    def j = 0
                    header.each { h ->
                        cell(j, 0, h).bold()
                        j++
                    }

                    def i = 1
                    list.each { l ->
                        cell(0, i, l[surnameFieldName] ?: "")
                        cell(1, i, l[nameFieldName] ?: "")
                        cell(2, i, l.username ?: "")
                        cell(3, i, l.email ?: "")

                        i++
                    }
                }
            }
        }
        return ftmp
    }


    // =====================================================================
    // Utility methods
    // =====================================================================

    private String lookupUserClassName() {
        SpringSecurityUtils.securityConfig.userLookup.userDomainClassName
    }

    private Class<?> lookupUserClass() {
        grailsApplication.getDomainClass(lookupUserClassName()).clazz
    }

    private String lookupRoleClassName() {
        SpringSecurityUtils.securityConfig.authority.className
    }

    private Class<?> lookupRoleClass() {
        grailsApplication.getDomainClass(lookupRoleClassName()).clazz
    }

    private String lookupUserRoleClassName() {
        SpringSecurityUtils.securityConfig.userLookup.authorityJoinClassName
    }

    private Class<?> lookupUserRoleClass() {
        grailsApplication.getDomainClass(lookupUserRoleClassName()).clazz
    }
}
