package org.gualdi.grails.plugins.cms

import grails.converters.JSON
import org.codehaus.groovy.grails.plugins.springsecurity.SpringSecurityUtils
import org.gualdi.grails.plugins.cms.exceptions.*
import org.gualdi.grails.plugins.cms.co.CmsRolesCommand
import org.gualdi.grails.plugins.cms.co.CmsUsersCommand
import org.gualdi.grails.plugins.cms.co.CmsUsersRolesCommand

import javax.servlet.ServletContext

class UsersControllerMixin extends AbstractUsersControllerMixin {

    def cmsBaseService
    def cmsSecurityService
    def cmsBaseUsersService
    def springSecurityService

    static menuEntry = [
            key: "cms.menu.users",
            order: 60,
            useSitetag: true,
            items: [
                    [key: 'cms.menu.users.manageRoles', action: 'listRoles'],
                    [key: 'cms.menu.users.manageUsers', action: 'listUsers']
            ]
    ]

    // =====================================================================
    // Roles
    // =====================================================================

    def listRoles() {
        def sitetag = params.sitetag
        def roles = cmsBaseUsersService.listRoles(params)

        render view: "/cms/roles/list", model: [list: roles]
    }

    def listAllRoles() {
        def sitetag = params.sitetag
        def excludeAdmins = params?.excludeAdmins == "true"
        def roles = cmsBaseUsersService.listAllRoles(params.q, excludeAdmins)

		def jsonMap = roles.collect {r -> return [id: r.id, authority: r.authority]}
		
		render jsonMap as JSON
    }

    def addRole() {
        def sitetag = params.sitetag

        CmsRolesCommand cmd = new CmsRolesCommand()

        if (request.post) {
            bindData(cmd, params)
            if (cmd.validate()) {
                cmsBaseUsersService.saveRole(cmd, sitetag)

                flash.message = "cms.roles.message.roleSaved"
                redirect action: "listRoles", params: [sitetag: sitetag]
            }
            else {
                render view: "/cms/roles/add", model: [cmd: cmd]
            }
        }
        else {
            render view: "/cms/roles/add", model: [cmd: cmd]
        }
    }

    def editRole() {
        def sitetag = params.sitetag

        CmsRolesCommand cmd = new CmsRolesCommand()

        if (request.post) {
            bindData(cmd, params)
            if (cmd.validate()) {
                cmsBaseUsersService.updateRole(cmd, sitetag)

                flash.message = "cms.roles.message.roleUpdated"
                redirect action: "listRoles", params: [sitetag: sitetag]
            }
            else {
                render view: "/cms/roles/edit", model: [cmd: cmd]
            }
        }
        else {
            def role = cmsBaseUsersService.getRole(params.long("id"))
            bindData(cmd, role.properties)
            cmd.id = role.id
            render view: "/cms/roles/edit", model: [cmd: cmd]
        }
    }

    def deleteRole() {
        def sitetag = params.sitetag
        def id = params.long("id")

        try {
            cmsBaseUsersService.deleteRole(id, sitetag)
            flash.message = "cms.roles.message.roleDeleted"
        }
        catch (CannotDeleteRoleException cdre) {
            flash.error = "cms.roles.error.cannotDeleteRole"                
        }

        redirect action: "listRoles", params: [sitetag: sitetag]
    }

    def cancelRole() {
        redirect action: "listRoles", params: [sitetag: params.sitetag]
    }

    // =====================================================================
    // Users
    // =====================================================================

    def listUsers() {
        def sitetag = params.sitetag
        def users = cmsBaseUsersService.listUsers(params)

        render view: "/cms/users/list", model: [list: users]
    }

    def listAllUsers() {
        def sitetag = params.sitetag
        def users = cmsBaseUsersService.listAllUsers(params.q)
		
		def jsonMap = users.collect {u -> return [id: u.id, name: u.name, surname: u.surname, username: u.username]}
		
		render jsonMap as JSON
    }

    def addUser() {
        def sitetag = params.sitetag

        CmsUsersCommand cmd = new CmsUsersCommand()

        def allRoles = cmsBaseUsersService.listAllRoles()

        if (request.post) {
            bindData(cmd, params)
            if (cmd.validate()) {
                cmsBaseUsersService.saveUser(cmd, sitetag)

                flash.message = "cms.users.message.userSaved"
                redirect action: "listUsers", params: [sitetag: sitetag]
            }
            else {
                render view: "/cms/users/add", model: [cmd: cmd, allRoles: allRoles]
            }
        }
        else {
            render view: "/cms/users/add", model: [cmd: cmd, allRoles: allRoles]
        }
    }

    def editUser() {
        def sitetag = params.sitetag

        CmsUsersCommand cmd = new CmsUsersCommand()

        def allRoles = cmsBaseUsersService.listAllRoles()

        if (request.post) {
            bindData(cmd, params)
            if (cmd.validate()) {
                cmsBaseUsersService.updateUser(cmd, sitetag)

                flash.message = "cms.users.message.userUpdated"
                redirect action: "listUsers", params: [sitetag: sitetag]
            }
            else {
                render view: "/cms/users/edit", model: [cmd: cmd, allRoles: allRoles]
            }
        }
        else {
            def user = cmsBaseUsersService.getUser(params.long("id"))
            bindData(cmd, user.properties, [exclude: ['password']])
            cmd.id = user.id
            cmd.roles = user.authorities.authority
            render view: "/cms/users/edit", model: [cmd: cmd, allRoles: allRoles]
        }
    }

    def deleteUser() {
        def sitetag = params.sitetag
        def id = params.long("id")

        try {
            cmsBaseUsersService.deleteUser(id, sitetag)
            flash.message = "cms.users.message.userDeleted"
        }
        catch (CannotDeleteUserException cdre) {
            flash.error = "cms.users.error.cannotDeleteUser"                
        }

        redirect action: "listUsers", params: [sitetag: sitetag]
    }

    def cancelUser() {
        redirect action: "listUsers", params: [sitetag: params.sitetag]
    }

    def assignUsers() {
        convertSelect2Params(['roles', 'users'])

        def sitetag = params.sitetag

        CmsUsersRolesCommand cmd = new CmsUsersRolesCommand()

        if (request.post) {
            bindData(cmd, params)
            if (cmd.validate()) {
                cmsBaseUsersService.assignRolesToUsers(cmd.roles, cmd.users, sitetag)

                flash.message = "cms.users.message.rolesAssigned"
                redirect action: "assignUsers", params: [sitetag: sitetag]
            }
            else {
                render view: "/cms/users/assign", model: [cmd: cmd]
            }
        }
        else {
            render view: "/cms/users/assign", model: [cmd: cmd]
        }
    }

    private convertSelect2Params(fields, separator = ",") {
        fields.each { f ->
            def val = params[f]
            if (val) {
                params[f] = val.tokenize(separator)
            }
        }
    }

    def exportUsers() {
        def sitetag = params.sitetag

        try {
            File file = cmsBaseUsersService.exportUsers()
            if (file) {
                def downloadFilename = "users.xls"
                sendFile(file, downloadFilename)
                file.delete()
                return null
            }
            else {
                flash.message = "cms.users.message.export.noData"
                redirect action: "listUsers", params: [sitetag: sitetag]
            }
        }
        catch (Exception dee) {
            flash.error = "cms.users.error.export.error"
            redirect action: "listUsers", params: [sitetag: sitetag]
        }
    }

    // Utils
    private sendFile(File file, String downloadFilename) {
        ServletContext context = getServletContext()

        response.setHeader("Content-Type", "application/force-download")
        response.setHeader("Content-Disposition", "attachment; filename=\"${downloadFilename}\"")
        response.setHeader("Content-Length", "${file.size()}")
        response.setHeader("Content-Transfer-Encoding", "Binary");

        def os = response.outputStream

        byte[] buff = null
        BufferedInputStream bis = new BufferedInputStream(new FileInputStream(file))
        try {
            buff = new byte[2048]
            int bytesRead = 0
            while ((bytesRead = bis.read(buff, 0, buff.size())) != -1) {
                os.write(buff, 0, bytesRead)
            }
        }
        finally {
            bis.close()
            os.flush()
            os.close()
        }
    }
}