package org.gualdi.grails.plugins.cms

import grails.plugin.asyncmail.AsynchronousMailMessage
import grails.plugin.jxl.builder.ExcelBuilder
import jxl.Sheet
import jxl.Workbook
import jxl.read.biff.BiffException
import org.gualdi.grails.plugins.cms.domain.*
import org.gualdi.grails.plugins.cms.enums.NewsletterStatus
import org.hibernate.Criteria
import org.springframework.transaction.annotation.Transactional

class CmsBaseNewslettersService {

    static transactional = false

    def sessionFactory
    def grailsApplication
    def propertyInstanceMap = org.codehaus.groovy.grails.plugins.DomainClassGrailsPlugin.PROPERTY_INSTANCE_MAP

    def mailService
    def asynchronousMailService
    def groovyPagesTemplateEngine

    // =====================================================================
    // Newsletters groups
    // =====================================================================

    NewsletterGroup getNewsletterGroup(Long id) {
        NewsletterGroup.get(id)
    }

    def listNewsletterGroups(String sitetag, Map params) {
        Site site = Site.findByTag(sitetag)

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ? params.offset.toInteger() : 0

        // select g.site_id, g.id, count(g.id)
        // from newsletter_group g, newsletter_subscription_group sg
        // where sg.group_id = g.id
        // group by g.id
        // order by g.site_id;

        // select g.site_id, g.id, g.name, count(sg.subscription_id)
        // from newsletter_group g LEFT JOIN newsletter_subscription_group sg on sg.group_id = g.id
        // where g.site_id = 2
        // group by g.id
        // order by g.site_id, g.name;

        // FIXME: se un gruppo con contiene subscriptions non viene visualizzato
        // Colleziona tutti i gruppi ed il conteggio di subscriptions per ogni gruppo
        /*
        String selectClause = "select distinct g, count(sg.subscription)"
        String fromClause = "from NewsletterGroup g, NewsletterSubscriptionGroup sg WHERE g.site = :site AND sg.group = g GROUP BY g ORDER BY g.name"
        String hql = selectClause + " " + fromClause
        String hqlCount = "select count(g) from NewsletterGroup g WHERE g.site = :site"
        def groupsList = NewsletterGroup.executeQuery(hql, [site: site], [max: params.max, offset: params.offset])

        def totalCount = NewsletterGroup.executeQuery(hqlCount, [site: site])

        def result = [:]
        result.items = groupsList
        result.totalCount = totalCount[0]
        */

        def query = """
        select g.id, g.tag, g.name, count(sg.subscription_id) as cnt
        from newsletter_group g LEFT JOIN newsletter_subscription_group sg on sg.group_id = g.id
        where g.site_id = ${site.id}
        group by g.id
        order by g.site_id, g.name
        """

        def qry = sessionFactory.currentSession.createSQLQuery(query)
                .setResultTransformer(Criteria.ALIAS_TO_ENTITY_MAP)
        if (params.offset) {
            qry.setFirstResult(params.offset)
        }
        if (params.max) {
            qry.setMaxResults(params.max)
        }

        def result = [:]
        result.items = qry.list()
        result.totalCount = NewsletterGroup.countBySite(site)

        return result
    }

    def listAllNewslettersGroups(String sitetag) {
        Site site = Site.findByTag(sitetag)

        def c = NewsletterGroup.createCriteria()
        def list = c.list {
            eq("site", site)
            order("name", "asc")
        }

        return list
    }

    @Transactional
    def saveNewsletterGroup(cmd, String sitetag) {
        Site site = Site.findByTag(sitetag)

        def group = new NewsletterGroup(cmd.properties)
        group.site = site
        group.save()
    }

    @Transactional
    def updateNewsletterGroup(cmd, String sitetag) {
        def group = getNewsletterGroup(cmd.id)
        group.properties = cmd.properties
        group.save()
    }

    @Transactional
    def deleteNewsletterGroup(Long id, String sitetag) {
        // TODO: add deletion checks
        def group = getNewsletterGroup(id)
        group.delete()
    }

    Boolean isNewsletterGroupTagUnique(String tag, String sitetag, Long id) {
        Boolean result = true
        if (tag) {
            Site site = Site.findByTag(sitetag)
            def groups = NewsletterGroup.findAllBySite(site)
            if (groups.any { g -> (g.tag == tag) && (id ? (g.id != id) : true) }) {
                result = false
            }
        }
        return result
    }

    // =====================================================================
    // Newsletters templates
    // =====================================================================

    NewsletterTemplate getNewsletterTemplate(Long id) {
        NewsletterTemplate.get(id)
    }

    def listNewsletterTemplates(String sitetag, Map params) {
        Site site = Site.findByTag(sitetag)

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def c = NewsletterTemplate.createCriteria()
        def list = c.list(max: params.max, offset: params.offset) {
            eq("site", site)
            order("name", "asc")
        }

        return list
    }

    def listAllNewslettersTemplates(String sitetag) {
        Site site = Site.findByTag(sitetag)

        def c = NewsletterTemplate.createCriteria()
        def list = c.list {
            eq("site", site)
            order("name", "asc")
        }

        return list
    }

    @Transactional
    def saveNewsletterTemplate(cmd, String sitetag) {
        Site site = Site.findByTag(sitetag)

        def template = new NewsletterTemplate(cmd.properties)
        template.site = site
        template.save()
    }

    @Transactional
    def updateNewsletterTemplate(cmd, String sitetag) {
        def template = getNewsletterTemplate(cmd.id)
        template.properties = cmd.properties
        template.save()
    }

    @Transactional
    def deleteNewsletterTemplate(Long id, String sitetag) {
        // TODO: add deletion checks
        def template = getNewsletterTemplate(id)
        template.delete()
    }

    // =====================================================================
    // Newsletters
    // =====================================================================

    Newsletter getNewsletter(Long id) {
        Newsletter.get(id)
    }

    String getCompiledNewsletter(Long id) {
        return compileNewsletterTemplate(id)
    }

    def listNewsletters(String sitetag, Map params) {
        Site site = Site.findByTag(sitetag)

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def c = Newsletter.createCriteria()
        def list = c.list(max: params.max, offset: params.offset) {
            eq("site", site)
            order("date", "desc")
            order("status", "desc")
        }

        return list
    }

    def listSentNewsletters(String sitetag, Map params) {
        Site site = Site.findByTag(sitetag)

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def c = Newsletter.createCriteria()
        def list = c.list(max: params.max, offset: params.offset) {
            and {
              eq("site", site)
              eq("status", NewsletterStatus.SENT)
            }
            order("date", "desc")
            order("status", "desc")
        }

        return list
    }

    @Transactional
    def saveNewsletter(cmd, String sitetag) {
        Site site = Site.findByTag(sitetag)

        def newsletter = new Newsletter(cmd.properties)
        newsletter.site = site
        newsletter.status = NewsletterStatus.EDITING
        newsletter.save(flush: true)

        def group
        cmd?.recipientsList?.each { g ->
            group = NewsletterGroup.get(g as Long)
            NewsletterRecipient.create newsletter, group, true
        }
    }

    @Transactional
    def updateNewsletter(cmd, String sitetag) {
        def newsletter = getNewsletter(cmd.id)
        newsletter.properties = cmd.properties
        newsletter.save(flush: true)

        // Update recipients
        if (cmd?.recipientsList) {
            def newRecipients = cmd?.recipientsList - newsletter?.recipients?.id

            // Remove recipients
            newsletter?.recipients?.each { r ->
                if (!(r.id in cmd.recipientsList)) {
                    NewsletterRecipient.remove newsletter, r, true
                }
            }

            // Add new recipients
            def group
            newRecipients.each { r ->
                group = NewsletterGroup.get(r as Long)
                NewsletterRecipient.create newsletter, group, true
            }
        }
        else {
            // Remove all recipients for this newsletter
            NewsletterRecipient.removeAll(newsletter)
        }
    }

    @Transactional
    def deleteNewsletter(Long id, String sitetag) {
        // TODO: add deletion checks
        def newsletter = getNewsletter(id)
        newsletter.delete()
    }

    def sendForTest(Long id) {
        def tmplBody = compileNewsletterTemplate(id)

        runAsync {
            try {
                Newsletter.withTransaction {
                    def newsletter = Newsletter.get(id)

                    def senderEmail = newsletter.sender
                    def recipientEmail = newsletter.testAddress

                    mailService.sendMail {
                        to recipientEmail
                        from senderEmail
                        subject newsletter.title
                        html tmplBody
                    }
                }
            }
            catch (Exception e) {
                log.error e
            }
        }
    }

    @Transactional
    def sendForReal(Long id) {
        def tmplBody = compileNewsletterTemplate(id)

        def newsletter = Newsletter.get(id)
        newsletter.status = NewsletterStatus.SENDING
        newsletter.save(flush: true)

        runAsync {
            try {
                Newsletter.withTransaction {
                    newsletter = Newsletter.get(id)

                    def senderEmail = newsletter.sender

                    // select distinct r.newsletter_id, sg.subscription_id
                    // from newsletter_recipient r, newsletter_group g, newsletter_subscription_group sg
                    // where r.newsletter_id = 43 and r.group_id = g.id and sg.group_id = r.group_id;

                    // Colleziona tutti gli indirizzi unici a cui inviare i messaggi
                    /*
                    String selectClause = "select distinct sg.subscription"
                    String fromClause = "from NewsletterRecipient r, NewsletterGroup g, NewsletterSubscriptionGroup sg WHERE r.newsletter = :newsletter AND r.group = g AND sg.group = r.group order by sg.subscription.email"
                    String hql = selectClause + " " + fromClause

                    def sendList = Newsletter.executeQuery(hql, [newsletter: newsletter])
                    */
                    String hql = """
                        select s
                        from NewsletterSubscription s
                        where exists (
                            select 1
                            from NewsletterSubscriptionGroup sg, NewsletterRecipient r
                            where sg.subscription = s
                            and sg.group = r.group
                            and r.newsletter = :newsletter
                        )
                        order by s.email
                        """
                    def sendList = NewsletterSubscription.executeQuery(hql, [newsletter: newsletter])

                    sendList.each { s ->
                        // Invia a tutti gli utenti collezionati
                        if (log.isDebugEnabled()) {
                            log.debug "Invio a -> ${s.name} ${s.surname} <${s.email}>"
                        }

                        try {
                            asynchronousMailService.sendMail {
                                to s.email
                                from senderEmail
                                subject newsletter.title
                                html tmplBody

                                jobId id

                                delete true
                            }
                        }
                        catch (Exception e1) {
                            log.error "Cannot send email to ${s.name} ${s.surname} <${s.email}>", e1
                        }
                    }

                    // Riavvia il job di invio perche' se i messaggi sono pochi (< 100??) non vengono inviati
                    try {
                        asynchronousMailService.sendImmediately()
                    }
                    catch (Exception e2 ) {
                        log.error e2
                    }

                    // Marca la newsletter come inviata
                    newsletter.dateSent = new Date()
                    newsletter.status = NewsletterStatus.SENT
                    newsletter.save(flush: true)
                }
            }
            catch (Exception e) {
                log.error e
            }

        }
    }

    def restartQueue() {
        asynchronousMailService.sendImmediately()
    }

    def calculateRecipientsNumber(Long newsletterId) {
        def newsletter = Newsletter.get(newsletterId)

        // FIXME - improve query - use count()
        String selectClause = "select distinct sg.subscription"
        String countClause = "count (sg.subscription)"
        String fromClause = "from NewsletterRecipient r, NewsletterGroup g, NewsletterSubscriptionGroup sg WHERE r.newsletter = :newsletter AND r.group = g AND sg.group = r.group"
        String hql = selectClause + " " + fromClause

        def sendList = Newsletter.executeQuery(hql, [newsletter: newsletter])

        return sendList.size()
    }

    private String compileNewsletterTemplate(Long newsletterId) {
        def newsletter = Newsletter.get(newsletterId)
        if (newsletter) {
            def templateCompiled = new StringWriter()

            def templateText = newsletter.template.code
            def tmpl = groovyPagesTemplateEngine.createTemplate(templateText, "template-${newsletter.template.id}")
            tmpl.make([newsletter: newsletter]).writeTo(templateCompiled)

            return templateCompiled.toString()
        }
        else {
            return null
        }
    }

    // =====================================================================
    // Subscriptions
    // =====================================================================

    NewsletterSubscription getSubscription(Long id) {
        NewsletterSubscription.get(id)
    }

    NewsletterSubscription getSubscription(String email) {
        NewsletterSubscription.findByEmail(email)
    }

    def listSubscriptions(String sitetag, Map params) {
        Site site = Site.findByTag(sitetag)

        def q = params.q
        q = "%" + q + "%"

        params.max = Math.min(params.max ? params.max.toInteger() : 10, 100)
        params.offset = params.offset ?: 0

        def c = NewsletterSubscription.createCriteria()
        def list = c.list(max: params.max, offset: params.offset) {
            eq("site", site)

            or {
                if (params.q) {
                    ilike('email', q)
                    ilike('name', q)
                    ilike('surname', q)
                }
            }

            order("surname", "asc")
            order("name", "asc")
            order("email", "asc")
        }

        return list
    }

    @Transactional
    def deleteSubscription(Long id, String sitetag) {
        // TODO: add deletion checks
        def subscription = getSubscription(id)
        subscription.delete()
    }

    File exportSubscriptions(String sitetag, String[] groups) {
        Site site = Site.findByTag(sitetag)

        def list

        if (groups) {
            def finalGroups = []
            NewsletterGroup group
            groups?.each { groupId ->
                group = NewsletterGroup.get(groupId)
                if (group) {
                    finalGroups << group
                }
            }
            try {
                def u = NewsletterSubscriptionGroup.createCriteria()
                list = u.listDistinct {
                    projections {
                        property("subscription")
                    }

                    eq("site", site)

                    or {
                        finalGroups.each { g ->
                            eq("group", g)
                        }
                    }

                    /*
                    subscription {
                        order("surname", "asc")
                        order("name", "asc")
                        order("email", "asc")
                    }
                    */
                }
            }
            catch (Exception e) {
                log.error e
            }
        }
        else {
            def u = NewsletterSubscription.createCriteria()
            list = u.list {
                eq("site", site)
                order("surname", "asc")
                order("name", "asc")
                order("email", "asc")
            }
        }

        File ftmp = null
        if (list.size() > 0) {
            ftmp = File.createTempFile("export-", ".xls")
            ftmp.deleteOnExit() //in case something goes wrong and you don't catch it

            def header = [
                    'SURNAME', 'NAME', 'EMAIL',
            ]

            new ExcelBuilder().workbook(ftmp.absolutePath) {
                sheet('Users') {

                    def j = 0
                    header.each { h ->
                        cell(j, 0, h).bold()
                        j++
                    }

                    def i = 1
                    list.each { l ->
                        cell(0, i, l.surname ?: "")
                        cell(1, i, l.name ?: "")
                        cell(2, i, l.email ?: "")

                        i++
                    }
                }
            }
        }
        return ftmp
    }

    @Transactional
    Map importSubscriptions(File inputFile, String sitetag, String[] groups) {
        def updated = 0
        def created = 0
        def errors = 0
        def duplicated = 0

        def startTime = System.currentTimeMillis()

        Site site = Site.findByTag(sitetag)
        Long siteId = site.id

        // Main group must exists. All addresses are always added to it
        NewsletterGroup mainGroup = NewsletterGroup.findBySiteAndTag(site, "main")
        if (!mainGroup) {
            mainGroup = new NewsletterGroup(site: site, tag: "main", name: "Main").save(flush: true)
        }

        def finalGroups = []

        // Always add to the main group
        finalGroups << mainGroup

        NewsletterGroup group
        groups?.each { groupId ->
            group = NewsletterGroup.get(groupId)
            if (group) {
                finalGroups << group
            }
        }

        def alreadyImported = [:]
        Workbook w
        try {
            w = Workbook.getWorkbook(inputFile)

            // Get the first sheet
            Sheet sheet = w.getSheet(0)

            String name = null
            String surname = null
            String email = null
            NewsletterSubscription subscription
            int counter = 0;
            for (int i = 1; i < sheet.getRows(); i++) {
                surname = sheet.getCell(0, i).getContents()
                name = sheet.getCell(1, i).getContents()
                email = sheet.getCell(2, i).getContents()
                if (email) {
                    if (!alreadyImported[email]) {

                        // Reload the site to avoid hibernate detaching in case of cleanup gorm session
                        site = Site.load(siteId)

                        alreadyImported[email] = true

                        def invalidSubscription = false
                        def addedOrUpdated = false
                        subscription = NewsletterSubscription.findBySiteAndEmail(site, email)
                        if (!subscription) {
                            subscription = new NewsletterSubscription(site: site, name: name, surname: surname, email: email).save(flush: true)
                            if (subscription) {
                                log.debug "NEW subscription for address ${email}"
                                created++
                                addedOrUpdated = true
                            }
                            else {
                                errors++
                                invalidSubscription = true
                            }
                        }
                        else {
                            if ((subscription.name != name) || (subscription.surname != surname)) {
                                subscription.name = name
                                subscription.surname = surname
                                subscription.save(flush: true)
                                if (subscription.hasErrors()) {
                                    errors++
                                    invalidSubscription = true
                                }
                                else {
                                    log.debug "UPDATE subscription for address ${email}"
                                    updated++
                                    addedOrUpdated = true
                                }
                            }
                        }

                        if (!invalidSubscription) {
                            // Create the associations with the required groups
                            finalGroups.each { grp ->
                                if (!NewsletterSubscriptionGroup.get(site.id, subscription.id, grp.id)) {

                                    NewsletterSubscriptionGroup.create site, subscription, grp, true
                                }
                            }
                        }
                        else {
                            log.error "INVALID address ${email}"
                        }
                    }
                    else {
                        log.debug "SKIPPING duplicate address ${email}"
                        duplicated++
                    }
                }

                if ((counter++ % 50) == 0) {
                    log.debug "GORM cleanup at ${counter}"
                    cleanUpGorm()
                }
            }
        }
        catch (BiffException e) {
            log.error e
        }

        def endTime = System.currentTimeMillis()
        def totalTime = (endTime - startTime) / 1000

        return [created: created, updated: updated, duplicated: duplicated, errors: errors, totalTime: totalTime]
    }

    @Transactional
    Map deleteSubscriptions(File inputFile, String sitetag) {
        def deleted = 0
        def errors = 0
        def duplicated = 0

        def startTime = System.currentTimeMillis()

        Site site = Site.findByTag(sitetag)

        def alreadyDeleted = [:]

        Workbook w
        try {
            w = Workbook.getWorkbook(inputFile)

            // Get the first sheet
            Sheet sheet = w.getSheet(0)

            String email = null
            NewsletterSubscription subscription
            int counter = 0;
            for (int i = 1; i < sheet.getRows(); i++) {
                email = sheet.getCell(2, i).getContents()

                if (email) {
                    if (!alreadyDeleted[email]) {

                        alreadyDeleted[email] = true

                        subscription = NewsletterSubscription.findBySiteAndEmail(site, email)
                        if (subscription) {
                            subscription.delete()
                            deleted++
                        }
                        else {
                            log.debug "CANNOT DELETE ${email}"
                            errors++
                        }
                    }
                    else {
                        log.debug "SKIPPING duplicate address ${email}"
                        duplicated++
                    }
                }
            }
        }
        catch (BiffException e) {
            log.error e
        }

        def endTime = System.currentTimeMillis()
        def totalTime = (endTime - startTime) / 1000

        return [deleted: deleted, duplicated: duplicated, errors: errors, totalTime: totalTime]
    }

    private cleanUpGorm() {
        def session = sessionFactory.currentSession
        session.flush()
        session.clear()
        propertyInstanceMap.get().clear()
    }

    // =====================================================================
    // Subscriptions workflow
    // =====================================================================

    @Transactional
    NewsletterSubscriptionCode subscribeUser(String sitetag, String name, String surname, String email, List<String> groups) {
        def groupsTags = groups?.join(',')
        def registrationCode = new NewsletterSubscriptionCode(
                sitetag: sitetag,
                name: name,
                surname: surname,
                email: email,
                groupsTags: groupsTags
        ).save(flush: true)

        return registrationCode
    }

    @Transactional
    NewsletterSubscription verifySubscription(String token, Boolean createGroups = false) {
        def subscriptionCode = token ? NewsletterSubscriptionCode.findByToken(token) : null
        if (!subscriptionCode) {
            return null
        }

        NewsletterSubscription subscription = confirmSubscribedUser(subscriptionCode, createGroups)

        subscriptionCode.delete()

        return subscription
    }

    @Transactional
    NewsletterSubscription confirmSubscribedUser(NewsletterSubscriptionCode subscriptionCode, Boolean createGroups = false) {
        Site site = Site.findByTag(subscriptionCode.sitetag)
        List<String> groups = subscriptionCode.groupsTags?.split(/,/)
        
        return subscribeUserDirect(site, subscriptionCode.name, subscriptionCode.surname, subscriptionCode.email, groups, createGroups)
    }
    
    @Transactional
    NewsletterSubscription subscribeUserDirect(String sitetag, String name, String surname, String email, List<String> groups, Boolean createGroups = false) {
        Site site = Site.findByTag(sitetag)
        return subscribeUserDirect(site, name, surname, email, groups, createGroups)
    }

    NewsletterSubscription subscribeUserDirect(Site site, String name, String surname, String email, List<String> groups, Boolean createGroups = false) {
        NewsletterGroup mainGroup = NewsletterGroup.findBySiteAndTag(site, "main")
        if (!mainGroup) {
            mainGroup = new NewsletterGroup(site: site, tag: "main", name: "Main").save(flush: true)
        }

        NewsletterSubscription subscription = NewsletterSubscription.findBySiteAndEmail(site, email)
        if (!subscription) {
            subscription = new NewsletterSubscription(site: site, name: name, surname: surname, email: email).save(flush: true)
        }

        NewsletterGroup group
        def finalGroups = []

        // Always add to the main group
        finalGroups << mainGroup
        groups?.each { groupTag ->
            group = NewsletterGroup.findBySiteAndTag(site, groupTag)
            if (createGroups && !group) {
                group = new NewsletterGroup(site: site, tag: groupTag, name: groupTag.capitalize()).save(flush: true)
            }

            if (group) {
                finalGroups << group
            }
        }

        // Create the associations with the required groups
        finalGroups.each { grp ->
            if (!NewsletterSubscriptionGroup.get(site.id, subscription.id, grp.id)) {
                NewsletterSubscriptionGroup.create site, subscription, grp, true
            }
        }

        return subscription
    }

    @Transactional
    NewsletterSubscriptionCode unsubscribeUser(String sitetag, String email) {
        def registrationCode = new NewsletterSubscriptionCode(
                sitetag: sitetag,
                email: email
        ).save(flush: true)

        return registrationCode
    }

    @Transactional
    boolean verifyUnsubscription(String token) {
        def subscriptionCode = token ? NewsletterSubscriptionCode.findByToken(token) : null
        if (!subscriptionCode) {
            return false
        }

        def result = confirmUnsubscribedUser(subscriptionCode)

        subscriptionCode.delete()

        return result
    }

    @Transactional
    boolean confirmUnsubscribedUser(NewsletterSubscriptionCode subscriptionCode) {
        Site site = Site.findByTag(subscriptionCode.sitetag)

        def result = false
        NewsletterSubscription subscription = NewsletterSubscription.findBySiteAndEmail(site, subscriptionCode.email)
        if (subscription) {
            deleteSubscription(subscription.id, subscriptionCode.sitetag)
            result = true
        }

        return result
    }

    // =====================================================================
    // Mail queue
    // =====================================================================

    List getMailQueue(String sitetag) {
        def site = Site.findByTag(sitetag, [readOnly: true])

        def query = "select n.id, n.title, n.date, count(m.id) from AsynchronousMailMessage m, Newsletter n where m.jobId = n.id and n.site = :site group by n.id order by n.date asc"
        def queue = AsynchronousMailMessage.executeQuery(query, [site: site], [readOnly: true])

        def result = []
        queue.each { itm ->
            result << [title: itm[1], date: itm[2], count: itm[3]]
        }

        return result
    }

    // =====================================================================
    // Subscriptions maintenance
    // =====================================================================

    @Transactional
    void cleanupExpiredSubscriptions() {
        log.debug "cleanupExpiredSubscription() start"
        def yesterday = new Date().minus(1)
        NewsletterSubscriptionCode.executeUpdate("delete from NewsletterSubscriptionCode c where c.dateCreated < :yesterday", [yesterday: yesterday])
        log.debug "cleanupExpiredSubscription() end"
    }
}
